<?php

namespace App\Http\Controllers;

use App\Models\Folder;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class FolderController extends Controller
{


    public function index(Request $request)
    {
        $folderId = $request->input('folder_id');

        if ($folderId) {
            $theFolder = Folder::with('children', 'assets', 'users')->findOrFail($folderId);

            $folders = $theFolder->children()
                ->orderByRaw('position IS NULL, position ASC')
                ->orderBy('created_at')
                ->get();
            $assets = $theFolder->assets()
                ->orderByRaw('position IS NULL, position ASC')
                ->orderBy('created_at')
                ->get();

            $path = $theFolder->getPath();
            $backUrl = route('admin.folders.index', ['folder_id=' . $theFolder->parent_id]);
        } else {
            $folders = Folder::with('children', 'assets', 'users')
                ->whereNull('parent_id')
                ->orderByRaw('position IS NULL, position ASC')
                ->orderBy('created_at')
                ->get();

            $assets = new Collection();

            $path = 'Home';
            $backUrl = null;
        }


        return view('admin.folders.index', compact('folders', 'assets', 'path', 'backUrl'));
    }

    public function updateOrder(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'folder_orders' => 'required|array|min:1',
                'folder_orders.*.folder_id' => 'required|integer|exists:folders,id',
                'folder_orders.*.position' => 'required|integer|min:0'
            ]);

            DB::beginTransaction();

            $folderOrders = $request->input('folder_orders', []);

            foreach ($folderOrders as $order) {
                $folderId = $order['folder_id'];
                $position = $order['position'];

                $folder = Folder::find($folderId);
                if ($folder) {
                    $folder->update([
                        'position' => $position
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Folder orders updated successfully'
            ]);

        } catch (ValidationException $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Validation error: ' . $e->getMessage(),
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error updating folder orders: ' . $e->getMessage()
            ], 500);
        }
    }


    public function create(Request $request)
    {
        $parentId = $request->query('parent_id');

        $parentFolder = null;
        if ($parentId) {
            $parentFolder = Folder::on()->find($parentId);
        }

        return view('admin.folders.create', compact('parentFolder'));
    }


    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:255',
            'parent_id' => 'nullable|exists:folders,id',
        ]);

        Folder::create($request->all());

        return redirect()->route('admin.folders.index', ['folder_id=' . $request->input('parent_id')])->with('success', 'Folder created successfully!');
    }


    public function confirmDelete(Folder $folder)
    {
        $title = "Are you sure you want to delete the folder <strong>$folder->name</strong>?";
        $desc = 'This action cannot be undone. Any sub-folders will also be deleted.';
        $deleteRoute = route('admin.folders.destroy', $folder);

        return view('admin.folders.delete', compact( 'title', 'desc', 'deleteRoute'));
    }


    public function destroy(Folder $folder)
    {
        $folderName = $folder->name;
        $folder->delete();

        return redirect()->route('admin.folders.index')->with('success', "Folder '{$folderName}' deleted successfully!");
    }


    public function showGrantForm(Request $request)
    {
        $folderId = $request->query('folder_id');


        if (!$folderId) {
            return redirect()->back()->withErrors([
                'Folder' => 'The folder can\'t be empty'
            ]);
        }

        $folder = Folder::on()->find($folderId);

        return view('admin.folders.grant', compact('folder'));
    }


    public function grantAccess(Request $request)
    {
        $request->validate([
            'folder_id' => 'required|exists:folders,id',
            'telegram_ids' => 'required|string',
        ]);

        $folder = Folder::find($request->input('folder_id'));

        $telegramIds = preg_split('/[\s,]+/', $request->input('telegram_ids'));
        $telegramIds = array_filter(array_map('trim', $telegramIds));

        if (empty($telegramIds)) {
            return back()->with('error', 'No valid Telegram IDs were provided.');
        }

        $users = User::query()
            ->whereIn('telegram_id', $telegramIds)
            ->orWhereIn('username', $telegramIds)
            ->get();


        $foundIds = $users->map(function ($user) {
            return [$user->telegram_id, $user->username];
        })->flatten()->filter()->toArray();


        $userIdsToAttach = $users->pluck('id')->toArray();
        $folder->users()->syncWithoutDetaching($userIdsToAttach);


        $notFoundIds = array_diff($telegramIds, $foundIds);
        $successMessage = count($userIdsToAttach) . ' user(s) were successfully added to the folder: "' . $folder->name . '".';

        $feedback = redirect()->back()->with('success', $successMessage);

        if (!empty($notFoundIds)) {
            $feedback->with('info', 'The following Telegram IDs were not found: ' . implode(', ', $notFoundIds));
        }

        return $feedback;
    }


    public function showRevokeForm(Request $request)
    {
        $folderId = $request->query('folder_id');


        if (!$folderId) {
            return redirect()->back()->withErrors([
                'Folder' => 'The folder can\'t be empty'
            ]);
        }

        $folder = Folder::on()->find($folderId);

        if ($folder->users->isEmpty()) {
            return redirect()->back()->withErrors(['No users have access to the folder!']);
        }

        return view('admin.folders.revoke', compact('folder'));
    }


    public function revokeAccess(Request $request)
    {
        $request->validate([
            'folder_id' => 'required|exists:folders,id',
            'telegram_ids' => 'required|string',
        ]);

        $folder = Folder::find($request->input('folder_id'));

        $telegramIds = preg_split('/[\s,]+/', $request->input('telegram_ids'));
        $telegramIds = array_filter(array_map('trim', $telegramIds));

        if (empty($telegramIds)) {
            return back()->with('error', 'No valid Telegram IDs were provided.');
        }

        $users = User::query()
            ->whereIn('telegram_id', $telegramIds)
            ->orWhereIn('username', $telegramIds)
            ->get();


        $foundIds = $users->map(function ($user) {
            return [$user->telegram_id, strtolower($user->username)];
        })->flatten()->filter()->toArray();


        $userIdsToRevoke = $users->pluck('id')->toArray();
        $folder->users()->detach($userIdsToRevoke);


        $notFoundIds = array_diff($telegramIds, $foundIds);

        $successMessage = count($userIdsToRevoke) . ' user(s) were successfully revoked access to the folder: "' . $folder->name . '".';

        $feedback = redirect()->route('admin.folders.index')->with('success', $successMessage);

        if (!empty($notFoundIds)) {
            $feedback->with('info', 'The following Telegram IDs were not found: ' . implode(', ', $notFoundIds));
        }

        return $feedback;
    }
}
